; ASSEMBLER ASIDE 1.14
;          title     "N8VEM CP/M BIOS, Version 0.9"
;          page      49 
;          .Z80
ENDFIL:    EQU       0                   ; FILL TOTAL BIOS AREA
HARDDRV:   EQU       1                   ; ENABLE HARD DRIVE CODE
FLOPDRV:   EQU       1                   ; ENABLE FLOPPY DRIVE CODE
DEBUG:     EQU       1                   ; ENABLE DEBUGGING CODE
FIX01:     EQU       0                   ; ENABLE DIRECT BIOS DISK CALLS 
                                         ; FROM LOW TPA
;**************************************************************
;*
;*        C B I O S  F O R N8VEM SBC'S
;*
;*        BY ANDREW LYNCH, WITH INPUT FROM MANY SOURCES
;*        MODIFIED BY JAMES MOXHAM AND BILL BEECH, NJ7P
;*        VERSION 0.1 16 JAN 09 - CLEANUP/STANDARDIZATION
;*        VERSION 0.2 28 JAN 09 - DPB IN ROM/RAM IMAGES
;*        VERSION 0.3 8 FEB 09 - DON'T RELOAD BIOS EVER
;*                  LOAD CCP_BDOS ONLY ON COLD BOOT
;*        VERSION 0.8 APR 09 - SET BAUD IAW JUMPERS 
;*        VERSION 0.9 25 AUG 09 - INTEGRATE THE FDC CODE
;*                  FIX THE IDE CODE
;*
;**************************************************************

; CPM ORIGIN CALCULATE

NK:      EQU     59             ;SYSTEM SIZE
BASE:    EQU     (NK-20)*1024  ;BIAS VALUE
CCPO:    EQU     BASE+3400H     ;BASE OF CCP
BDOSO:   EQU     BASE+3C00H     ;BASE OF BDOS
BIOSO:   EQU     BASE+4A00H     ;BASE OF BIOS

          ORG       BIOSO               ;ORIGIN OF THIS PROGRAM
                    
;         LOW MEMORY ADDRESSES

IOBYTE:   EQU       00003H              ;CPM I/O BYTE
CDISK:    EQU       00004H              ;CURRENT DISK 0=A,...,15=P
BUFFER:   EQU       00080H              ;INITIAL BUFFER ADDRESS

;         CHARACTER CONSTANTS

ENDS:     EQU       0FFH
CR:       EQU       00DH
LF:       EQU       00AH
CTLZ:     EQU       01AH

;         HARDWARE AND PORT ADDRESSES

PIO:      EQU       060H                ;BASE IO ADDRESS OF PIA
UART:     EQU       068H                ;BASE IO ADDRESS OF UART
RAMPAG:   EQU       078H                ;IO ADDRESS OF RAM PAGE LATCH
ROMPAG:   EQU       07CH                ;IO ADDRESS OF ROM PAGE LATCH

CPMROM:   EQU       00500H              ;WHERECP/M IS STORED IN ROM
CCPSIZ:   EQU       00800H              ;CCP LENGTH
CPMSIZ:   EQU       01600H              ;CPM LENGTH
STACK:    EQU       0FFFFH              ;LOCAL STACK POINTER

DPBLEN:   EQU       15                  ;DPB LENGTH

          IF        HARDDRV
;IDE REGISTER       IO PORT             ;FUNCTION

IDELO:    EQU       020H                ;DATA PORT (LOW BYTE)
IDEERR:   EQU       021H                ;READ: ERROR REGISTER;WRITE: PRECOMP
IDESECTC: EQU       022H                ;SECTOR COUNT
IDESECTN: EQU       023H                ;SECTOR NUMBER
IDECYLLO: EQU       024H                ;CYLINDER LOW
IDECYLHI: EQU       025H                ;CYLINDER HIGH
IDEHEAD: EQU        026H                ;DRIVE/HEAD
IDESTTS: EQU        027H                ;READ: STATUS;WRITE: COMMAND
IDEHI:    EQU       028H                ;DATA PORT (HIGH BYTE)
IDECTRL: EQU        02EH                ;READ: ALTERNATIVE STATUS;WRITE;DEVICE CONTROL
IDEADDR: EQU        02FH                ;DRIVE ADDRESS (READ ONLY)
          ENDIF

          IF        FLOPDRV
FMSR:     EQU       036H                ; ADDRESS OF MAIN STATUS REGISTER
FDATA:    EQU       037H                ; FLOPPY DATA REGISTER
FLATCH:   EQU       03AH                ; FLOPPY CONFIGURATION LATCH
FDMA:     EQU       03CH                ; PSEUDO DMA ADDRESS

; FDC CONFIGURATION LATCH OUTPUT BIT PATTERNS
MOTOR:    EQU       00000000B ; BIT PATTERN IN LATCH FOR MOTOR CONTROL (ON)
TERMCN:   EQU       00000001B ; BIT PATTERN IN LATCH TO WRITE A TC STROBE
RESETL:   EQU       00000010B ; BIT PATTERN IN LATCH TO RESET ALL BITS
MINI:     EQU       00000100B ; BIT PATTERN IN LATCH TO SET MINI MODE FDC9229 LOW DENS=1, HIGH DENS=0
PRECOMP: EQU        00100000B ; BIT PATTERN IN LATCH TO SET WRITE PRECOMP 125 NS:
FDDENSITY: EQU      01000000B ; BIT PATTERN IN LATCH TO FLOPPY LOW DENSITY (HIGH IS 0)
FDREADY: EQU        10000000B ; BIT PATTERN IN LATCH TO FLOPPY READY (P-34):
          ENDIF

;---------------------------------------------------------

;         JUMP VECTOR TABLE FOR INDIVIDUAL SUBROUTINES

          JP        BOOT                ;COLD START
WBOOTE:   JP        WBOOT               ;WARM START
          JP        CONST               ;CONSOLE STATUS
          JP        CONIN               ;CONSOLE CHARACTER IN
          JP        CONOUT              ;CONSOLE CHARACTER OUT
          JP        LISTX                ;LIST CHARACTER OUT
          JP        PUNCH               ;PUNCH CHARACTER OUT
          JP        READER              ;READER CHARACTER OUT
          JP        HOME                ;HOME DISK
          JP        SELDSK              ;SELECT DISK
          JP        SETTRK              ;SET TRACK NUMBER
          JP        SETSEC              ;SET SECTOR NUMBER
          JP        SETDMA              ;SET DMA ADDRESS
          JP        READ                ;READ DISK
          JP        WRITE               ;WRITE DISK
          JP        LISTST              ;RETURN LIST STATUS
          JP        SECTRN              ;SECTOR TRANSLATE
          JP        FORMAT              ;FORMAT A TRACK
                    
;-----------------------------------------------------------------------------

          IF        DEBUG     
DUMPF:    DEFB        0                   ;DUMP FLAG
          ENDIF
          
;-----------------------------------------------------------------------------

;         FIXED DATA TABLES FOR ALL DRIVES
;         0= RAMDISK, 1=ROMDISK, 
;         2=HDPART2, 3=HDPART3, AND 4=HDPART4

TRANO:    EQU       0                   ;SECTOR TRANSLATION TABLE OFFSET (0 IF NO TRANS)
TMP0O:    EQU       2
TMP1O:    EQU       4
TMP2O:    EQU       6
DIRBO:    EQU       8                   ;128 BYTE DIRECTORY BUFFER OFFSET (SHARED BY ALL DPH'S)
DPBO:     EQU       10                  ;DISK PARAMETER BLOCK OFFSET (CAN BE SHARED)
CHKO:     EQU       12                  ;CHECK OFFSET (UNIQUE FOR FIXED MEDIA)
ALLVO:    EQU       14                  ;ALLOCATION VECTOR OFFSET (UNIQUE TO EACH DEVICE)


DPHBASE:  

;         DISK PARAMETER HEADER FOR DISK 00
DPH0:     DEFW        0,0
          DEFW        0,0
          DEFW        DIRBF,DPBLK1
          DEFW        CHK01,ALL01

;         DISK PARAMETER HEADER FOR DISK 01
DPH1:     DEFW        0,0
          DEFW        0,0
          DEFW        DIRBF,DPBLK0
          DEFW        CHK00,ALL00

          IF        HARDDRV
;         DISK PARAMETER HEADER FOR DISK 02
DPH2:     DEFW        0,0
          DEFW        0,0
          DEFW        DIRBF,DPBLK2
          DEFW        CHK02,ALL02
          
;         DISK PARAMETER HEADER FOR DISK 03
DPH3:     DEFW        0,0
          DEFW        0,0
          DEFW        DIRBF,DPBLK2
          DEFW        CHK03,ALL03
          
;         DISK PARAMETER HEADER FOR DISK 04
DPH4:     DEFW        0,0
          DEFW        0,0
          DEFW        DIRBF,DPBLK2
          DEFW        CHK04,ALL04
          
;         DISK PARAMETER HEADER FOR DISK 05
DPH5:     DEFW        0,0
          DEFW        0,0
          DEFW        DIRBF,DPBLK2
          DEFW        CHK05,ALL05
          ENDIF

          IF        FLOPDRV
;         DISK PARAMETER HEADER FOR DISK 06
DPH5:     DEFW        0,0
          DEFW        0,0
          DEFW        DIRBF,DPBLK2
          DEFW        CHK06,ALL06
          ENDIF

SPTO:     EQU       0                   ;SECTORS PER TRACK OFFSET
BSFO:     EQU       2                   ;BLOCK SHIFT FACTOR OFFSET
BSMO:     EQU       3                   ;BLOCK SHIFT MASK OFFSET
EXTO:     EQU       4                   ;EXTENT MASK OFFSET
DSMO:     EQU       5                   ;DISK STORAGE MAXIMUM BLOCK OFFSET
DRMO:     EQU       7                   ;DIRECTORY ENTRY MAXIMUM OFFSET
AL0O:     EQU       9                   ;ALLOCATION BLOCK 0 OFFSET
AL01:     EQU       10                  ;ALLOCATION BLOCK 1 OFFSET
CKSO:     EQU       11                  ;DIRECTORY CHECK VECTOR SIZE OFFSET
OFFO:     EQU       13                  ;RESERVE TRACKS COUNT OFFSET

;         ROMDISK
DPBLK0:   DEFW        0                   ;FILLED BY SELDSK
          DEFB        0
          DEFB        0
          DEFB        0
          DEFW        0
          DEFW        0
          DEFB        0
          DEFB        0
          DEFW        0
          DEFW        0         

;         RAMDISK
DPBLK1:   DEFW        0                   ;FILLED BY SELDSK
          DEFB        0
          DEFB        0
          DEFB        0
          DEFW        0
          DEFW        0
          DEFB        0
          DEFB        0
          DEFW        0
          DEFW        0
          
          IF        HARDDRV
;         IDE HARD DISK 8MB
DPBLK2:   DEFW        256                 ;256 SECTORS OF 128 BYTES PER 32K TRACK
          DEFB        5                   ;BLOCK SHIFT FACTOR (SIZE OF ALLOCATION BLOCK)
          DEFB        31                  ;PART OF THE ALLOCATION BLOCK SIZE MATH
          DEFB        1                   ;DEFINES SIZE OF EXTENT (DIRECTORY INFO)
          DEFW        2041                ;BLOCKSIZE [4096] * NUMBER OF BLOCKS + 1 = DRIVE SIZE
                                        ;HD PARTITION 2 IS 16384 SECTORS LONG
                                        ;AT 512 BYTES EACH WHICH IS 
                                        ;2041 BLOCKS AT 4096 BYTES A PIECE.
          DEFW        511                 ;NUMBER OF DIRECTORY ENTRIES
          DEFB        11110000B ;BIT MAP OF SPACE ALLOCATED TO DIRECTORY
          DEFB        00000000B ;DIRECTORY CAN HAVE UP TO 16 BLOCKS ALLOCATED
          DEFW        0                   ;SIZE OF DIRECTORY CHECK [0 IF NON REMOVEABLE]
          DEFW        1                   ;1 TRACK (32K) RESERVED FOR SYSTEM
          ENDIF

          IF        FLOPDRV
;         FLOPPY DISK 720KB
DPBLK3:
          DEFW  36                        ; 36 SECTORS OF 128 BYTES PER 4.5K TRACK
          DEFB  4                         ; BLOCK SHIFT FACTOR (SIZE OF ALLOCATION BLOCK)
          DEFB  15                        ; PART OF THE ALLOCATION BLOCK SIZE MATH
          DEFB  0                         ; DEFINES SIZE OF EXTENT (DIRECTORY INFO)
          DEFW  350                       ; BLOCKSIZE [2048] * NUMBER OF BLOCKS + 1 = DRIVE SIZE
          DEFW  127                       ; NUMBER OF DIRECTORY ENTRIES
          DEFB  11000000B                 ; BIT MAP OF SPACE ALLOCATED TO DIRECTORY
          DEFB  00000000B                 ; DIRECTORY CAN HAVE UP TO 16 BLOCKS ALLOCATED
          DEFW  32                        ; SIZE OF DIRECTORY CHECK [0 #IF NON REMOVEABLE]
          DEFW  4                         ; FIRST 4 TRACKS TRACKS RESERVED (18K FOR SYSTEM)
                                        ; SYSTEM IS ROM LOADER, CCP, BDOS, CBIOS, AND MONITOR
;         FLOPPY DISK 360KB
DPBLK3:
          DEFW  36                        ; 36 SECTORS OF 128 BYTES PER 4.5K TRACK
          DEFB  4                         ; BLOCK SHIFT FACTOR (SIZE OF ALLOCATION BLOCK)
          DEFB  15                        ; PART OF THE ALLOCATION BLOCK SIZE MATH
          DEFB  1                         ; DEFINES SIZE OF EXTENT (DIRECTORY INFO)
          DEFW  181                       ; BLOCKSIZE [2048] * NUMBER OF BLOCKS + 1 = DRIVE SIZE
          DEFW  127                       ; NUMBER OF DIRECTORY ENTRIES
          DEFB  11110000B                 ; BIT MAP OF SPACE ALLOCATED TO DIRECTORY
          DEFB  00000000B                 ; DIRECTORY CAN HAVE UP TO 16 BLOCKS ALLOCATED
          DEFW  32                        ; SIZE OF DIRECTORY CHECK [0 #IF NON REMOVEABLE]
          DEFW  2                         ; FIRST 4 TRACKS TRACKS RESERVED (18K FOR SYSTEM)
                                        ; SYSTEM IS ROM LOADER, CCP, BDOS, CBIOS, AND MONITOR
          ENDIF

;-----------------------------------------------------------------------------

;         INDIVIDUAL SUBROUTINES TO PERFORM EACH FUNCTION

;         COLD BOOT ENTRY

BOOT:     DI                            ;DISABLE INTERRUPT
          LD        A,080H              ;SWITCH OUT ROM, BRING IN RAM PAGE
          OUT       (ROMPAG),A
          XOR       A                   ;BRING IN LOWEST 32K RAM PAGE
          OUT       (RAMPAG),A
          LD        (IOBYTE),A          ;CLEAR THE IOBYTE
          LD        (CDISK),A ;SELECT DISK ZERO
          LD        (WRMFLG),A          ;SET COLD BOOT.
          LD        SP,BUFFER ;USE SPACE BELOW BUFFER FOR STACK
          CALL      PIOINI              ;INITIALIZE PIO
          CALL      UARTINI             ;INITIALIZE UART
          LD        HL,TXT_COLD         ;PRINT COLD STARTUP MESSAGE
          CALL      PRTMSG
          JR        GOCPM               ;COMMON INITIALIZATION
          
;-----------------------------------------------------------------------------

;         WARM BOOT ENTRY

WBOOT:    DI                            ;DISABLE INTERRUPT
          XOR       A                   ;SWITCH IN LOWER 32K ROM PAGE
          OUT       (ROMPAG),A
          LD        HL,CPMROM ;LOAD CCP+BDOS FROM ROM
          LD        DE,CCPO
          LD        BC,CPMSIZ
          LDIR
          LD        A,080H              ;SWITCH OUT ROM, BRING IN RAM PAGE
          OUT       (ROMPAG),A
          XOR       A                   ;BRING IN LOWEST 32K RAM PAGE
          OUT       (RAMPAG),A
          LD        SP,BUFFER ;USE SPACE BELOW BUFFER FOR STACK
          LD        A,0FFH              ;SET COLD BOOT.
          LD        (WRMFLG),A
          LD        HL,TXT_WARM         ;PRINT WARM STARTUP MESSAGE
          CALL      PRTMSG

;-----------------------------------------------------------------------------

;         COMMON INITIALIZATION AND GO TO CP/M

GOCPM:    IM        1                   ;SET INTERRUPT MODE 1
          EI                            ;ENABLE INTERRUPTS
          IF        HARDDRV
          CALL      IDE_SOFT_RESET      ;RESET THE IDE HARD DISK
          ENDIF
          LD        HL,TXT_SO ;PRINT REST OF STARTUP MESSAGE
          CALL      PRTMSG
          LD        A,0C3H              ;SET JMP TO WBOOT
          LD        (0),A
          LD        HL,WBOOTE
          LD        (1),HL
          LD        (5),A               ;SET BDOS ENTRY POINT
          LD        HL,BDOSO+6          ;ACTUAL ENTRY POINT
          LD        (6),HL
          LD        BC,80H              ;DEFAULT DMA ADDRESS IS 80H
          CALL      SETDMA
          LD        A,(CDISK) ;GET CURRENT DISK NUMBER
          LD        C,A                 ;SEND TO THE CCP
          LD        A,(WRMFLG)          ;CLEAR THE CCP BUFFER?
          OR        A
          JP        NZ,CCPO+3 ;YES
          JP        CCPO                ;GO TO CP/M FOR FURTHER PROCESSING
          
;-----------------------------------------------------------------------------

;         SIMPLE CHARACTER I/O HANDLERS

;         INITIALIZE UART

;         TABLE OF BAUD RATE DIVISORS (1.832 OSC)
BRTAB:    DEFB        96                  ;1200
          DEFB        48                  ;2400
          DEFB        24                  ;4800
          DEFB        12                  ;9600
          DEFB        06                  ;19200
          DEFB        03                  ;38400
          DEFB        01                  ;115200

UARTINI:LD          A,080H
          OUT       (UART+3),A
          IN        A,(PIO+1)           ;GET JUMPER SETTINGS
          RRA
          RRA
          RRA
          RRA
          RRA
          LD        HL,BRTAB
          LD        DE,0
          LD        E,A
          ADD       HL,DE
          LD        A,(HL)
          OUT       (UART),A
          XOR       A
          OUT       (UART+1),A
          LD        A,003H
          OUT       (UART+3),A          
          RET
          
;         CONSOLE STATUS, RETURN 0FFH IF CHARACTER READY, 00H IF NOT

CONST:    IN        A,(UART + 5)        ;READ LINE STATUS REGISTER
          AND       001H                ;TEST IF DATA IN RECEIVE BUFFER
          RET       Z                   ;NO
          LD        A,0FFH              ;YES, PUT 0FFH IN A AND RETURN
          RET

;         CONSOLE INPUT, RETURN CHARACTER A

CONIN:    CALL      CONST               ;RETURNS A=0 OR A=0FFH
;         CP        0                   ;KEEP LOOPING WHILE IT IS A ZERO
          JR        Z,CONIN
          IN        A,(UART)  ;READ THE CHARACTER FROM THE UART
          AND       07FH                ;REMOVE PARITY      
          RET

;          CONSOLE CHARACTER OUTPUT FROM REGISTER C

CONOUT:   IN        A,(UART + 5)        ;READ LINE STATUS REGISTER
          AND       020H                ;TEST IF UART IS READY TO SEND
          JR        Z,CONOUT  ;IF NOT REPEAT
          LD        A,C                 ;GET TO ACCUMULATOR
          OUT       (UART),A  ;THEN WRITE THE CHAR TO UART
          RET

;         INITIALIZE PIO

PIOINI:   LD        A,0B2H              ;SET IT FOR DRIVING PRINTER
          OUT       (PIO+3),A
          RET

;         LIST CHARACTER FROM REGISTER C

LISTX:     RET

;         RETURN LIST STATUS (0 IF NOT READY, 0FFH IF READY)

LISTST:   XOR       A                   ;0 IS ALWAYS OK TO RETURN
          RET

;         PUNCH CHARACTER FROM REGISTER C

PUNCH:    RET

;         READ CHARACTER INTO REGISTER A FROM READER DEVICE

READER:   LD        A,CTLZ              ;ENTER END OF FILE FOR NOW 
          RET

;-----------------------------------------------------------------------------

;         DUMP THE DRIVE, TRACK, SECTOR, PAGE AND OFFSET OF THE CURRENT RECORD

          IF        DEBUG
DMPDKR:   LD        A,(DUMPF) ;DUMP ENABLED?
          OR        A
          RET       Z
          LD        HL,TXT_RED
          JR        DMPDK1
          
DMPDKW:   LD        A,(DUMPF) ;DUMP ENABLED?
          OR        A
          RET       Z         
          LD        HL,TXT_WRT
                              
DMPDK1:   PUSH      HL
          LD        HL,TXT_CL
          CALL      PRTMSG
          POP       HL
          CALL      PRTMSG
          LD        HL,TXT_DRV
          CALL      PRTMSG
          LD        A,(DRIVE)
          CALL      PUTBYT
          LD        HL,TXT_HED
          CALL      PRTMSG
          LD        A,(HEAD)
          CALL      PUTBYT
          LD        HL,TXT_TRK
          CALL      PRTMSG
          LD        HL,(TRACK)
          CALL      PUTADR
          LD        HL,TXT_SEC
          CALL      PRTMSG
          LD        HL,(SECTOR)
          CALL      PUTADR
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        HL,(LBA_OFFSET_LO)
          CALL      PUTADR
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        HL,(LBA_OFFSET_HI)
          CALL      PUTADR
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        HL,(LBA_TARGET_LO)
          CALL      PUTADR
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        HL,(LBA_TARGET_HI)
          CALL      PUTADR
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        A,(IDE_LBA0)
          CALL      PUTBYT
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        A,(IDE_LBA1)
          CALL      PUTBYT
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        A,(IDE_LBA2)
          CALL      PUTBYT
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        A,(IDE_LBA3)
          CALL      PUTBYT
          LD        HL,TXT_SPC
          CALL      PRTMSG
          LD        A,(SECTOR_INDEX)
          CALL      PUTBYT
;         LD        HL,TXT_PAG
;         CALL      PRTMSG
;         LD        A,(PAGE)
;         CALL      PUTBYT
;         LD        HL,TXT_ADR
;         CALL      PRTMSG
;         LD        HL,(OFFSET)
;         CALL      PUTADR
          RET

DMPDKS:   LD        A,(DUMPF) ;DUMP ENABLED?
          OR        A
          RET       Z
          LD        HL,TXT_SPC
          CALL      PRTMSG
          IN        A,(IDESTTS)
          CALL      PUTBYT
          RET
          
;-----------------------------------------------------------------------------

;         OUTPUT A HEX ADDRESS

PUTADR:   LD        A,H                 ;OUT HIGH BYTE
          CALL      PUTBYT
          LD        A,L                 ;OUT LOW BYTE
          CALL      PUTBYT
          RET

;-----------------------------------------------------------------------------

;         OUTPUT A HEX BYTE

PUTBYT:   PUSH      AF                  ;SAVE LOW NIBBLE
          RRCA                          ;GET HIGH NIBBLE
          RRCA
          RRCA
          RRCA
          CALL      PUTNIB              ;OUTPUT IT
          POP       AF
PUTNIB:   AND       0FH                 ;STRIP OFF GARBAGE
          ADD       A,'0'               ;MAKE IT PRINTABLE
          CP        ':'
          JR        C,PUTNIB1
          ADD       A,7
PUTNIB1:LD          C,A
          CALL      CONOUT
          RET
          ENDIF

;-----------------------------------------------------------------------------

;         PRINT OUT STRING ENDING WITH "END" CHARACTER

PRTMSG:   LD        A,(HL)              ;GET CHARACTER TO A
          CP        ENDS                 ;TEST FOR END BYTE
          RET       Z                   ;DONE
          LD        C,A                 ;PUT CHAR TO PRINT VALUE IN REG C FOR CONOUT
          CALL      CONOUT              ;SEND CHARACTER TO CONSOLE FROM REG C
          INC       HL                  ;INC POINTER, TO NEXT CHAR
          JR        PRTMSG              ;TRANSMIT LOOP
          
          
;-----------------------------------------------------------------------------

;         I/O DRIVERS FOR THE DISK FOLLOW

;         SELECT DISK GIVEN BY REGISTER C

SELDSK:   LD        HL,0                ;NULL IS ERROR
          LD        A,C
;         IF        HARDDRV
;         CP        5                   ;MUST BE BETWEEN 0 AND 4
;         ELSE
;         CP        2                   ;MUST BE BETWEEN 0 AND 1
;         ENDIF
          CP        6                   ;MUST BE BETWEEN 0 AND 6
          RET       NC                  ;NO CARRY IF 7,8...
          LD        (DRIVE),A
          LD        A,E                 ;SAVE LOGON REQUEST
          LD        (LOGREQ),A
          CALL      DPHCALC             ;DPH ADDRESS FOR GOOD RETURN
          CP        6                   ;FLOPPY DRIVE
          JP        Z,SETUPDRIVE
          CP        2                   ;DO NOTHING FOR HD0 FOR NOW
          RET       Z
          CP        3                   ;DO NOTHING FOR HD1 FOR NOW
          RET       Z
          CP        4                   ;DO NOTHING FOR HD2 FOR NOW
          RET       Z
          CP        5                   ;DO NOTHING FOR HD3 FOR NOW
          RET       Z
          LD        A,(LOGREQ)          ;ALREADY LOGGED ON?
          AND       001H
          RET       NZ                  ;YES
          DI
          PUSH      HL                  ;SAVE DPH ADDRESS
          IF        FIX01
          LD        HL,0                ;GET USER SP
          ADD       HL,SP
          LD        SP,STACK  ;SWITCH TO LOCAL STACK
          PUSH      HL                  ;SAVE USERS SP
          ENDIF
          CALL      DPBCALC             ;GET DPB ADDRESS
          LD        A,(DRIVE) ;GET DRIVE NUMBER
          OR        A                   ;RAM DISK
          JR        Z,SELRAM
          CP        1                   ;ROM DISK
          JR        Z,SELROM
          JR        SELCOM              ;HD OR FLOPPY

SELRAM:   LD        A,1                 ;SELECT RAM PAGE 1
          OUT       (RAMPAG),A          ;SEND TO PORT MAPPER
          LD        (PAGE),A  ;SAVE COPY (JUST BECAUSE)
          LD        HL,080H             ;DPB OFFSET IN RAM
                                        ;DE CONTAINS ADDRESS OF DPB
          LD        BC,DPBLEN ;DPB LENGTH
          LDIR
                    
          XOR       A                   ;SELECT RAM PAGE 0
          OUT       (RAMPAG),A          ;SEND TO PORT MAPPER
          LD        (PAGE),A  ;SAVE COPY (JUST BECAUSE)
          JR        SELCOM

SELROM:   XOR       A                   ;SELECT ROM PAGE 0
          OUT       (ROMPAG),A          ;SEND TO PORT MAPPER
          LD        HL,080H             ;DPB OFFSET IN ROM
                                        ;DE CONTAINS ADDRESS OF DPB
          LD        BC,DPBLEN ;DPB LENGTH
          LDIR
                    
          LD        A,080H              ;SWITCH OUT ROM 
          OUT       (ROMPAG),A          ;SEND TO PORT MAPPER
SELCOM:   
          IF        FIX01
          POP       HL                  ;GET USERS SP
          LD        SP,HL               ;RESTORE IT
          ENDIF     
          POP       HL                  ;RESTORE DPH ADDRESS
          EI
          RET
                    
;-----------------------------------------------------------------------------

;         GET DPH ADDRESS FOR CURRENT DRIVE INTO HL

DPHCALC:LD          A,(DRIVE) ;L=DISK NUMBER 0,1,2,3,4
          LD        L,A
          LD        H,0                 ;HIGH ORDER ZERO
          CALL      MUL16
          LD        DE,DPHBASE
          ADD       HL,DE               ;HL=DPBASE(DRIVE*16)
          LD        (DPHSTO),HL         ;SAVE IT
          RET
                    
;-----------------------------------------------------------------------------

;         GET DPB ADDRESS FOR CURRENT DRIVE INTO DE
                    
DPBCALC:LD          IX,(DPHSTO)         ;GET CURRENT DPH ADDRESS
          LD        E,(IX+DPBO)         ;GET DPB ADDRESS
          LD        D,(IX+DPBO+1)       
          LD        (DPBSTO),DE         ;SAVE IT
          RET                           
                    
;-----------------------------------------------------------------------------

;         CALCULATE PAGE AND OFFSET
;         SPT IS ALWAYS A POWER OF 2 FOR RAM/ROM DISKS!

CALCPO:   LD        HL,(TRACK)
          CALL      MUL16               ;SPT == 16!
          LD        BC,(SECTOR)
          ADD       HL,BC               ;LOGICAL SECTOR (H=PAGE, L*128=OFFSET)
          LD        A,(DRIVE) ;RAM DISK?
          OR        A
          JR        NZ,CALPO1 ;NO
          INC       H                   ;STEP PAST LOWER RAM PAGE
CALPO1:   LD        A,H                 ;SET ROM PAGE
          LD        (PAGE),A  ;SAVE COPY (JUST BECAUSE)
          LD        H,0                 ;CLEAR PAGE OVERFLOW
          CALL      MUL128
          LD        (OFFSET),HL         ;OFFSET
          RET

MUL128:   ADD       HL,HL
          ADD       HL,HL
MUL32:    ADD       HL,HL
MUL16:    ADD       HL,HL
MUL8:     ADD       HL,HL
          ADD       HL,HL
          ADD       HL,HL
          RET
          
;         RESET PAGE BACK TO RAM.  

RPAGE:    LD        A,080H              ;DESELECT ROM PAGE
          OUT       (ROMPAG),A
          XOR       A                   ;SELECT RAM PAGE 0
          OUT       (RAMPAG),A
          LD        (PAGE),A  ;SAVE COPY
          RET

;-----------------------------------------------------------------------------

;         MOVE TO THE TRACK 00 POSITION OF CURRENT DRIVE    
          
HOME:     LD        BC,0

;-----------------------------------------------------------------------------

;         SET TO TRACK IN BC

SETTRK:   LD        (TRACK),BC
          RET
          
;-----------------------------------------------------------------------------

;         SET SECTOR GIVEN BY REGISTER BC

SETSEC:   LD        (SECTOR),BC
          RET
                    
;-----------------------------------------------------------------------------

;         TRANSLATE THE SECTOR IN BC USING THE
;         TRANSLATE TABLE IN DE, RETURN PHYSICAL SECTOR IN HL

SECTRN:   LD        H,B
          LD        L,C
          RET

;-----------------------------------------------------------------------------

;         SET DMA ADDRESS GIVEN BY BC

SETDMA:   LD        (DMAADR),BC
          RET

;-----------------------------------------------------------------------------

;         FORMAT A TRACK 

FORMAT:   RET

;-----------------------------------------------------------------------------

READ:     DI                            ;DISABLE INTERRUPTS
          IF        FIX01
          LD        HL,0                ;GET USER SP
          ADD       HL,SP
          LD        SP,STACK  ;SWITCH TO LOCAL STACK
          PUSH      HL                  ;SAVE USERS SP
          ENDIF
          LD        A,(DRIVE) ;GET DRIVE
          CP        001H                ;FIND OUT WHICH DRIVE IS BEING REQUESTED
          JR        Z,READ_EEPROM_DISK ;ARE WE READING RAM OR ROM?
                                        ;READ FROM 22K EEPROM DISK
          CP        000H                ;
          JR        Z,READ_RAM_DISK     ;READ FROM 448K RAM DISK
          IF        HARDDRV
          CP        002H
          JR        Z,READ_HD01         ;READ FROM 8 MB IDE FIRST HARD DISK

          CP        003H
          JP        Z,READ_HD02         ;READ FROM 8 MB IDE FIRST HARD DISK

          CP        004H
          JP        Z,READ_HD03         ;READ FROM 8 MB IDE FIRST HARD DISK
          
          CP        005H
          JP        Z,READ_HD04         ;READ FROM 8 MB IDE FIRST HARD DISK
          ENDIF
          IF        FLOPDRV
          
          CP        005H
          JP        Z,FLOPPYREAD        ;READ FROM FLOPPY DISK
          ENDIF
READ_EEPROM_DISK:
          CALL      CALCPO              ;GET PAGE AND OFFSET
          OUT       (ROMPAG),A
                    
          JR        RDDSK1

READ_RAM_DISK:
          CALL      CALCPO              ;GET PAGE AND OFFSET
          OUT       (RAMPAG),A
                    
RDDSK1:

          LD        HL,(OFFSET)         ;GET ROM/RAM ADDRESS
          LD        DE,TMPBUF ;LOAD HL WITH TEMP BUF ADDRESS
          LD        BC,128
          LDIR

          CALL      RPAGE               ;SET PAGE TO CP/M RAM
                    
          LD        HL,TMPBUF ;GET ROM/RAM ADDRESS
          LD        DE,(DMAADR)         ;LOAD HL WITH DMA ADDRESS
          LD        BC,128
          LDIR

LE9C0H:
DSKCOM1:
TESTA:    EQU 0E9C0H-0EA52H             ; -146 dez




;TESTB:    EQU LE9C0H-LEA52H


          IF        FIX01
          POP       HL                  ;GET USERS SP
          LD        SP,HL               ;RESTORE IT
          ENDIF     
          XOR       A                   ;SET NO ERROR
          EI                            ;RE-ENABLE INTERRUPTS
          RET

          IF        HARDDRV
READ_HD01:
          LD        HL,00001H ;INITIALIZE LBA OFFSET SECTOR LO WORD
          JR        READHD
READ_HD02:
          LD        HL,04001H ;INITIALIZE LBA OFFSET SECTOR LO WORD
          JR        READHD
READ_HD03:
          LD        HL,08001H ;INITIALIZE LBA OFFSET SECTOR LO WORD
          JR        READHD
READ_HD04:
          LD        HL,0C001H ;INITIALIZE LBA OFFSET SECTOR LO WORD
READHD:   LD        (LBA_OFFSET_LO),HL

          LD        HL,00000H ;INITIALIZE LBA OFFSET SECTOR HI WORD
;         THIS ACTUALLY POINTS TO LOGICAL BLOCK 0 - THE PARTITION TABLE ON THE DRIVE!
          LD        (LBA_OFFSET_HI),HL

                                        ;BDOS TRACK PARAMETER (16 BITS)
                                        ;BDOS SECTOR PARAMETER (16 BITS)


          LD        HL,(TRACK)          ;LOAD TRACK # (WORD)
          LD        B,L                 ;SAVE LOWER 8 BITS (TRACK # 0-255)
          LD        HL,(SECTOR)         ;LOAD SECTOR # (WORD)
          LD        H,B                 ;HL IS 8 BIT TRACK IN H, 8 BIT SECTOR IN L


          CALL      CONVERT_IDE_SECTOR_CPM ;COMPUTE WHERE THE CP/M SECTOR IS ON THE
                                        ;IDE PARTITION

                                        ;MAP COMPUTED IDE HD SECTOR TO LBA REGISTERS

                                        ;LBA REGISTERS STORE 28 BIT VALUE OF IDE HD SECTOR ADDRESS

          LD        A,(LBA_TARGET_LO) ;LOAD LBA REGISTER 0 WITH SECTOR ADDRESS TO READ
          LD        (IDE_LBA0),A
          LD        A,(LBA_TARGET_LO+1) ;LOAD LBA REGISTER 1 WITH SECTOR ADDRESS TO READ
          LD        (IDE_LBA1),A
          LD        A,(LBA_TARGET_HI) ;LOAD LBA REGISTER 2 WITH SECTOR ADDRESS TO READ
          LD        (IDE_LBA2),A
          LD        A,(LBA_TARGET_HI+1) ;LOAD LBA REGISTER 3 WITH SECTOR ADDRESS TO READ
          AND       00001111B ;ONLY LOWER FOUR BITS ARE VALID
          ADD       A,11100000B         ;ENABLE LBA BITS 5:7=111 IN IDE_LBA3
          LD        (IDE_LBA3),A
          
          IF        DEBUG
          CALL      DMPDKR
          ENDIF

                                        ;READ IDE HD SECTOR

          CALL      IDE_READ_SECTOR     ;READ THE IDE HARD DISK SECTOR
          
          IF        DEBUG
          CALL      DMPDKS
          ENDIF

          
;         NEED TO ADD ERROR CHECKING HERE, CARRY FLAG IS SET IF IDE_READ_SECTOR SUCCESS

                                        ;COMPUTE STARTING ADDRESS OF CP/M SECTOR IN READ IDE HD 
                                        ; SECTOR BUFFER

          LD        HL,SECTOR_BUFFER ;LOAD HL WITH SECTOR BUFFER ADDRESS
          LD        A,(SECTOR_INDEX) ;GET THE SECTOR INDEX (CP/M SECTOR OFFSET IN BUFFER)
          RRCA                          ;MOVE BIT 0 TO BIT 7
          RRCA                          ;DO AGAIN - IN EFFECT MULTIPLY BY 64
          LD        D,0                 ;PUT RESULT AS 16 VALUE IN DE, UPPER BYTE IN D IS 0
          LD        E,A                 ;PUT ADDRESS OFFSET IN E
          ADD       HL,DE               ;MULTIPLY BY 2, TOTAL MULTIPLICATION IS X 128
          ADD       HL,DE               ;CP/M SECTOR STARTING ADDRESS IN IDE HD SECTOR BUFFER

                                        ;COPY CP/M SECTOR TO BDOS DMA ADDRESS BUFFER
          LD        DE,(DMAADR)         ;LOAD HL WITH DMA ADDRESS
          LD        BC,128
                    LDIR

          JR        DSKCOM1

          ENDIF

;-----------------------------------------------------------------------------

;  WRITE DISK

WRITE:    DI                            ;DISABLE INTERRUPTS
          IF        FIX01
          LD        HL,0                ;GET USER SP
          ADD       HL,SP
          LD        SP,STACK  ;SWITCH TO LOCAL STACK
          PUSH      HL                  ;SAVE USERS SP
          ENDIF

          LD        A,(DRIVE) ;GET DRIVE
          CP        001H                ;FIND OUT WHICH DRIVE IS BEING REQUESTED
          JR        Z,RDONLY  ;JUMP TO READ ONLY ROUTINE (CANT WRITE TO ROM)
                                        ;READ ONLY, FROM 22K EEPROM DISK, ERROR ON WRITE
          CP        000H                ;
          JR        Z,WRITE_RAM_DISK ;WRITE TO 448K RAM DISK
          IF        HARDDRV
          CP        002H
          JR        Z,WRITE_HD01        ;WRITE TO 8 MB IDE FIRST HARD DISK

          CP        003H
          JP        Z,WRITE_HD02        ;WRITE TO 8 MB IDE FIRST HARD DISK

          CP        004H
          JP        Z,WRITE_HD03        ;WRITE TO 8 MB IDE FIRST HARD DISK

          CP        005H
          JP        Z,WRITE_HD04        ;WRITE TO 8 MB IDE FIRST HARD DISK
          ENDIF
          IF        FLOPDRV
          
          CP        005H
          JP        Z,FLOPPYWRITE       ;WRITE TO FLOPPY DISK
          ENDIF

RDONLY:   LD        HL,TXT_RO ;SET HL TO START OF ERROR MESSAGE
          CALL      PRTMSG              ;PRINT ERROR MESSAGE
          LD        A,001H              ;SEND BAD SECTOR ERROR BACK

LEA52H:
;  OFFSET -126 BYTES OR +129 BYTES
;          JR        DSKCOM1
;
;           jr 0eac4h
;
          DEFB 18H
          DEFB 6CH
;
;
;
;          
WRITE_RAM_DISK:
          LD        HL,(DMAADR)         ;GET DMA ADDRESS
          LD        DE,TMPBUF ;LOAD HL WITH TEMP BUF ADDRESS
          LD        BC,128
          LDIR

          CALL      CALCPO              ;GET PAGE AND OFFSET
          OUT       (RAMPAG),A          ;SEND TO PORT MAPPER
          LD        HL,TMPBUF ;GET TEMP BUFFER ADDRESS
          LD        DE,(OFFSET)         ;LOAD HL WITH DMA ADDRESS (WHERE TO WRITE TO)
          LD        BC,128
          LDIR
                    
          CALL      RPAGE               ;SET BACK TO RAM
          XOR       A
          JP        DSKCOM1
          IF        HARDDRV
WRITE_HD01:
          LD        HL,00001H ;INITIALIZE LBA OFFSET SECTOR LO WORD
          JR        WRITHD
WRITE_HD02:
          LD        HL,04001H ;INITIALIZE LBA OFFSET SECTOR LO WORD
          JR        WRITHD
WRITE_HD03:
          LD        HL,08001H ;INITIALIZE LBA OFFSET SECTOR LO WORD
          JR        WRITHD
WRITE_HD04:
          LD        HL,0C001H ;INITIALIZE LBA OFFSET SECTOR LO WORD
WRITHD:   LD        (LBA_OFFSET_LO),HL

          LD        HL,00000H ;INITIALIZE LBA OFFSET SECTOR HI WORD
          LD        (LBA_OFFSET_HI),HL

                                        ;BDOS TRACK PARAMETER (16 BITS)
                                        ;BDOS SECTOR PARAMETER (16 BITS)

          LD        HL,(TRACK)          ;LOAD TRACK # (WORD)
          LD        B,L                 ;SAVE LOWER 8 BITS (TRACK # 0-255)
          LD        HL,(SECTOR)         ;LOAD SECTOR # (WORD)
          LD        H,B                 ;HL IS 8 BIT TRACK IN H, 8 BIT SECTOR IN L

          CALL      CONVERT_IDE_SECTOR_CPM ;COMPUTE WHERE THE CP/M SECTOR IS ON THE
                                        ;IDE PARTITION

                                        ;MAP COMPUTED IDE HD SECTOR TO LBA REGISTERS

                                        ;LBA REGISTERS STORE 28 BIT VALUE OF IDE HD SECTOR ADDRESS

          LD        A,(LBA_TARGET_LO) ;LOAD LBA REGISTER 0 WITH SECTOR ADDRESS TO READ
          LD        (IDE_LBA0),A
          LD        A,(LBA_TARGET_LO+1) ;LOAD LBA REGISTER 1 WITH SECTOR ADDRESS TO READ
          LD        (IDE_LBA1),A
          LD        A,(LBA_TARGET_HI) ;LOAD LBA REGISTER 2 WITH SECTOR ADDRESS TO READ
          LD        (IDE_LBA2),A
          LD        A,(LBA_TARGET_HI+1) ;LOAD LBA REGISTER 3 WITH SECTOR ADDRESS TO READ
          AND       00001111B ;ONLY LOWER FOUR BITS ARE VALID
          ADD       A,11100000B         ;ENABLE LBA BITS 5:7=111 IN IDE_LBA3
          LD        (IDE_LBA3),A
          
          IF        DEBUG
          CALL      DMPDKW
          ENDIF
          
                                        ;READ IDE HD SECTOR

          CALL      IDE_READ_SECTOR     ;READ THE IDE HARD DISK SECTOR

          IF        DEBUG
          CALL      DMPDKS
          ENDIF
          
;         NEED TO ADD ERROR CHECKING HERE, CARRY FLAG IS SET IF IDE_READ_SECTOR SUCCESS

                                        ;COMPUTE STARTING ADDRESS OF CP/M SECTOR IN READ IDE HD 
                                        ; SECTOR BUFFER

          LD        HL,SECTOR_BUFFER ;LOAD HL WITH SECTOR BUFFER ADDRESS
          LD        A,(SECTOR_INDEX) ;GET THE SECTOR INDEX (CP/M SECTOR OFFSET IN BUFFER)
          RRCA                          ;MOVE BIT 0 TO BIT 7
          RRCA                          ;DO AGAIN - IN EFFECT MULTIPLY BY 64
          LD        D,0                 ;PUT RESULT AS 16 VALUE IN DE, UPPER BYTE IN D IS 0
          LD        E,A                 ;PUT ADDRESS OFFSET IN E
          ADD       HL,DE               ;CP/M SECTOR STARTING ADDRESS IN IDE HD SECTOR BUFFER
          ADD       HL,DE               ;MULTIPLY BY 2, TOTAL MULTIPLICATION IS X 128

          LD        (OFFSET),HL         ;KEEP CP/M SECTOR ADDRESS FOR LATER USE

                                        ;COPY CP/M SECTOR FROM BDOS DMA ADDRESS BUFFER

          LD        HL,(DMAADR)         ;LOAD HL WITH DMA ADDRESS (WHERE THE DATA TO BE WRITTEN IS)
          LD        DE,(OFFSET)         ;LOAD CP/M SECTOR ADDRESS (WHERE THE DATA IS TO BE WRITTEN)
          LD        BC,128
          LDIR

                                        ;IDE HD SECTOR IS NOW UPDATED WITH CURRENT CP/M SECTOR DATA 
                                        ;SO WRITE TO DISK

          CALL      IDE_WRITE_SECTOR ;WRITE THE UPDATED IDE HARD DISK SECTOR

          IF        DEBUG
          CALL      DMPDKS
          ENDIF
          
;         NEED TO ADD ERROR CHECKING HERE, CARRY FLAG IS SET IF IDE_WRITE_SECTOR SUCCESS
          JP        DSKCOM1
          
;-----------------------------------------------------------------------------

CONVERT_IDE_SECTOR_CPM:                 ;COMPUTES WHERE THE CP/M SECTOR IS IN THE IDE PARTITION
                                        ;IDE HD SECTORS ARE 512 BYTES EACH, CP/M SECTORS ARE 128 
                                        ;BYTES EACH
                                        ;MAXIMUM SIZE OF CP/M DISK IS 8 MB = 65536 (16 BITS) X 128 
                                        ;BYTES PER SECTOR
                                        ;IDE HD PARTITION CAN HAVE AT MOST 16384 IDE SECTORS -> 65536 
                                        ;CP/M SECTORS
                                        ;EACH IDE HD SECTOR CONTAINS 4 ADJACENT CP/M SECTORS
                                        ;
                                        ;
                                        ;INPUT:
                                        ;IDE HD PARTITION STARTING SECTOR NUMBER (FROM PARTITION TABLE)
                                        ; - LOWER 16 BITS STORED IN LBA_OFFSET_LO
                                        ; - UPPER 16 BITS STORED IN LBA_OFFSET_HI
                                        ;PARTITION OFFSET IN HL (16 BITS)
                                        ; - A UNIQUELY COMPUTED FUNCTION BASED ON GEOMETRY OF 
                                        ;DISKS NUMBER OF
                                        ;   CP/M TRACKS AND SECTORS SPECIFIED IN DPB
                                        ;
                                        ;
                                        ;OUTPUT:
                                        ;IDE TARGET SECTOR (SENT TO IDE HD CONTROLLER FOR READ OPERATION)
                                        ; - LOWER 16 BITS STORED IN LBA_TARGET_LO
                                        ; - UPPER 16 BITS STORED IN LBA_TARGET_HI
                                        ;CP/M TO IDE HD SECTOR MAPPING PARAMETER STORED IN SECTOR_INDEX
                                        ; - 8 BIT VALUE WITH 4 LEGAL STATES (00, 01, 02, 04) WHICH IS
                                        ;   TO BE USED TO COMPUTE STARTING ADDRESS OF 128 BYTE CP/M 
                                        ;SECTOR ONCE
                                        ;   512 BYTE IDE HD SECTOR READ INTO MEMORY BUFFER
                                        ;

                                        ;ROTATE WITH CARRY 16 BIT TRACK,SECTOR VALUE IN HL TO GET 14 
                                        ;BIT IDE HD
                                        ;TARGET SECTOR IN PARTITION
                                        ;KEEP LAST TWO BITS IN B FOR IDE HD SECTOR TO CP/M SECTOR 
                                        ;TRANSLATION

                                        ;COMPUTE SECTOR_INDEX 

          XOR       A                   ;ZERO ACCUMULATOR
          LD        A,L                 ;STORE LAST 2 BITS OF L IN B
          AND       00000011B ;
          LD        B,A
          LD        (SECTOR_INDEX),A ;LOCATES WHERE THE 128 BYTE CP/M SECTOR
                                        ;IS WITHIN THE 512 BYTE IDE HD SECTOR

                                        ;COMPUTE WHICH IDE HD SECTOR TO READ TO WITHIN 4 CP/M SECTORS 
                                        ;SHIFTS 16 BIT PARTITION OFFSET TO THE RIGHT 2 BITS AND ADDS 
                                        ;RESULT TO
                                        ;IDE HD PARTITION STARTING SECTOR

                                        ;SHIFT PARTITION OFFSET RIGHT 1 BIT

          SCF
          CCF                           ;CLEAR CARRY FLAG
          LD        A,H                 ;16 BIT ROTATE HL WITH CARRY
          RRA
          LD        H,A                 ;ROTATE HL RIGHT 1 BIT (DIVIDE BY 2)
          LD        A,L
          RRA
          LD        L,A

                                        ;SHIFT PARTITION OFFSET RIGHT 1 BIT

          SCF
          CCF                           ;CLEAR CARRY FLAG
          LD        A,H                 ;16 BIT ROTATE HL WITH CARRY
          RRA
          LD        H,A                 ;ROTATE HL RIGHT 1 BIT (DIVIDE BY 2)
          LD        A,L
          RRA
          LD        L,A

                                        ;ADD RESULTING 14 BIT VALUE TO IDE HD PARTITION STARTING SECTOR
                                        ;STORE RESULT IN IDE HD TARGET SECTOR PARAMETER

          LD        A,(LBA_OFFSET_LO) ;16 BIT ADD OF LBA_OFFSET_LO WITH HL
          ADD       A,L
          LD        (LBA_TARGET_LO),A
          LD        A,(LBA_OFFSET_LO+1)
          ADC       A,H
          LD        (LBA_TARGET_LO+1),A ;STORE OVERFLOW BIT IN CARRY

          LD        HL,00000H
          LD        A,(LBA_OFFSET_HI) ;16 BIT ADD WITH CARRY OF LBA_OFFSET_HI WITH 0000
          ADC       A,L
          LD        (LBA_TARGET_HI),A
          LD        A,(LBA_OFFSET_HI+1)
          ADC       A,H
          LD        (LBA_TARGET_HI+1),A

          RET
          
;-----------------------------------------------------------------------------

IDE_READ_SECTOR:
          CALL      IDE_WAIT_BUSY_READY ;MAKE SURE DRIVE IS READY TO PROCEED
          RET       NC
          CALL      IDE_SETUP_LBA       ;TELL DRIVE WHAT SECTOR IS REQUIRED
          LD        A,020H
          OUT       (IDESTTS),A         ;020H = IDE 'READ SECTOR' COMMAND 

IDE_SREX:
          CALL      IDE_WAIT_BUSY_READY ;MAKE SURE DRIVE IS READY TO PROCEED
          RET       NC
          CALL      IDE_TEST_ERROR      ;ENSURE NO ERROR WAS REPORTED
          RET       NC
          CALL      IDE_WAIT_BUFFER     ;WAIT FOR FULL BUFFER SIGNAL FROM DRIVE
          RET       NC

          CALL      IDE_READ_BUFFER     ;GRAB THE 256 WORDS FROM THE BUFFER
          SCF                           ;CARRY = 1 ON RETURN = OPERATION OK
          RET

;-----------------------------------------------------------------------------

IDE_WRITE_SECTOR:
          CALL      IDE_WAIT_BUSY_READY ;MAKE SURE DRIVE IS READY TO PROCEED
          RET       NC
          CALL      IDE_SETUP_LBA       ;TELL DRIVE WHAT SECTOR IS REQUIRED
          LD        A,030H
          OUT       (IDESTTS),A         ;030H = IDE 'WRITE SECTOR' COMMAND 
          CALL      IDE_WAIT_BUSY_READY
          RET       NC
          CALL      IDE_TEST_ERROR      ;ENSURE NO ERROR WAS REPORTED
          RET       NC
          CALL      IDE_WAIT_BUFFER     ;WAIT FOR BUFFER READY SIGNAL FROM DRIVE
          RET       NC
          CALL      IDE_WRITE_BUFFER ;SEND 256 WORDS TO DRIVE'S BUFFER
          CALL      IDE_WAIT_BUSY_READY ;MAKE SURE DRIVE IS READY TO PROCEED
          RET       NC
          CALL      IDE_TEST_ERROR      ;ENSURE NO ERROR WAS REPORTED
          RET       NC
          SCF                           ;CARRY = 1 ON RETURN = OPERATION OK
          RET

;-----------------------------------------------------------------------------

IDE_SOFT_RESET:
;         LD        A,00000110B         ;NO INTERRUPTS, RESET DRIVE = 1
;         OUT       (IDECTRL),A
          LD        A,00000010B         ;NO INTERRUPTS, RESET DRIVE = 0
          OUT       (IDECTRL),A
          CALL      IDE_WAIT_BUSY_READY
          RET

;------------------------------------------------------------------------------
;IDE INTERNAL SUBROUTINES 
;------------------------------------------------------------------------------

IDE_WAIT_BUSY_READY:
          LD        DE,0
IDE_WBSY:
          LD        B,5
IDE_DLP:
          DJNZ      IDE_DLP
          INC       DE
          LD        A,D
          OR        E
          JR        Z,IDE_TO
          IN        A,(IDESTTS)         ;READ ERROR REG
          AND       11000000B ;MASK OFF BUSY AND RDY BITS
          XOR       01000000B ;WE WANT BUSY(7) TO BE 0 AND RDY(6) TO BE 1
          JR        NZ,IDE_WBSY
          SCF                           ;CARRY 1 = OK
          RET

IDE_TO:   XOR       A                   ;CARRY 0 = TIMED OUT
          RET

;----------------------------------------------------------------------------

IDE_TEST_ERROR:
          SCF
          IN        A,(IDESTTS)
          LD        B,A                 ;NEW
          AND       00000001B ;TEST ERROR BIT
          SCF                           ;NEW
          RET       Z

          LD        A,B                 ;NEW
          AND       00100000B
          SCF
          JR        NZ,IDE_ERR          ;TEST WRITE ERROR BIT

          IN        A,(IDEERR)          ;READ ERROR FLAGS

IDE_ERR:
          OR        A                   ;CARRY 0 = ERROR
          RET                           ;IF A = 0, IDE BUSY TIMED OUT

;-----------------------------------------------------------------------------

IDE_WAIT_BUFFER:
          LD        DE,0
IDE_WDRQ:
          LD        B,5
IDE_BLP:
          DJNZ    IDE_BLP
          INC       DE
          LD        A,D
          OR        E
          JR        Z,IDE_TO2
          IN        A,(IDESTTS)         ;WAIT FOR DRIVE'S 512 BYTE READ BUFFER 
          AND       00001000B ;TO FILL (OR READY TO FILL)
          JR        Z,IDE_WDRQ
          SCF                           ;CARRY 1 = OK
          RET

IDE_TO2:
          XOR       A                   ;CARRY 0 = TIMED OUT
          RET

;------------------------------------------------------------------------------

IDE_READ_BUFFER:
          PUSH      HL
          LD        HL,SECTOR_BUFFER
          LD        B,0                 ;256 WORDS (512 BYTES PER SECTOR)
IDEBUFRD:
          IN        A,(IDELO) ;LOW BYTE OF WORD FIRST       
          LD        (HL),A
          IN        A,(IDEHI) ;THEN HIGH BYTE OF WORD
          INC       HL
          LD        (HL),A
          INC       HL
          DEC       B
          JR        NZ,IDEBUFRD
          POP       HL
          RET

;-----------------------------------------------------------------------------

IDE_WRITE_BUFFER:
          PUSH      HL
          LD        HL,SECTOR_BUFFER
          LD        B,0                 ;256 WORDS (512 BYTES PER SECTOR)
IDEBUFWT:
          INC       HL
          LD        A,(HL)
          DEC       HL
          OUT       (IDEHI),A ;SET UP HIGH LATCHED BYTE BEFORE
          LD        A,(HL)
          OUT       (IDELO),A ;WRITING WORD WITH WRITE TO LOW BYTE
          INC       HL
          INC       HL
          DEC       B
          JR        NZ,IDEBUFWT
          POP       HL
          RET

;-----------------------------------------------------------------------------

IDE_SETUP_LBA:
          LD        A,1
          OUT       (IDESECTC),A        ;SET SECTOR COUNT = 1         
          LD        A,(IDE_LBA0)
          OUT       (IDESECTN),A        ;SET LBA 0:7
          LD        A,(IDE_LBA1)
          OUT       (IDECYLLO),A        ;SET LBA 8:15
          LD        A,(IDE_LBA2)
          OUT       (IDECYLHI),A        ;SET LBA 16:23
          LD        A,(IDE_LBA3)
          AND       00001111B ;LOWEST 4 BITS USED ONLY
          OR        11100000B ;TO ENABLE LBA MODE
          OUT       (IDEHEAD),A         ;SET LBA 24:27 + BITS 5:7=111
          RET
          ENDIF
          
          IF        FLOPDRV
;__SETUPDRIVE_______________________________________________________________________________ 
;
;         SETUP FLOPPY DRIVE SETTINGS 
;___________________________________________________________________________________
;
;
;
SETUPDRIVE:
          LD        A,RESETL            ; RESET SETTINGS
          OR        MINI                          ; SELECT MINI FLOPPY (low dens=1, high dens=0)
          OR        PRECOMP                       ; SELECT PRECOMP 
          OR        FDDENSITY           ; SELECT DENSITY
          OR        FDREADY                       ; SELECT READY SIGNAL
          LD        (FLATCH_STORE),A    ; SAVE SETTINGS
          LD        A,01H                         ;
          LD        (UNIT),A            ; SET UNIT 1
          LD        A,2                           ; DENSITY
          LD        (DENS),A            ;
          LD        A,09                          ;
          LD        (EOTSEC),A                    ; LAST SECTOR OF TRACK                            
          LD        A,7FH                         ;
          LD        (SRTHUT),A                    ; STEP RATE AND HEAD UNLOAD TIME
          LD        A,05H                         ;
          LD        (HLT),A                       ; HEAD LOAD TIME
          LD        A,0DH                         ;
          LD        (GAP),A                       ; GAP 
          LD        A,80H                         ;
          LD        (SECSIZ),A                    ; SECTOR SIZE /4
                                                  ;
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
                                                  ;                                                 
          LD        HL,FLATCH_STORE               ; POINT TO FLATCH
          RES       1,(HL)                        ; SET MOTOR ON
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER
          NOP                                     ;
          NOP                                     ;
          LD        A,00H                         ; ZERO TRACK
          LD        (TRACK),A           ; STORE TRACK
          CALL      SETTRACK            ; DO IT   
          NOP                                     ;
          NOP                                     ;
          LD        HL,FLATCH_STORE               ; POINT TO FLATCH
          SET       1,(HL)                        ; SET MOTOR OFF
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER
          RET
;
;__OUTFLATCH_______________________________________________________________________________ 
;
;         SEND SETTINGS TO FLOPPY CONTROLLER
;___________________________________________________________________________________
;
OUTFLATCH:
          LD        A,(FLATCH_STORE)    ; SET A TO SETTINGS
          OUT       (FLATCH),A                    ; OUTPUT TO CONTROLLER
          RET

                    
;__FLOPPYREAD_______________________________________________________________________________ 
;
;         READ A FLOPPY DISK SECTOR     
;___________________________________________________________________________________
;         
FLOPPYREAD:
          CALL      DMPDKR
          LD        A,46H                         ; BIT 6 SETS MFM, 06H IS READ COMMAND
          LD        (CMD),A
          JP        DSKOP
;
;__FLOPPYWRITE_______________________________________________________________________________ 
;
;         WRITE A FLOPPY DISK SECTOR    
;___________________________________________________________________________________
;         
FLOPPYWRITE:
          CALL      DMPDKW
          LD        A,45H                         ; BIT 6 SETS MFM, 05H IS WRITE COMMAND
          LD        (CMD),A
          JP        DSKOP
;
;__DSKOP_______________________________________________________________________________ 
;
;         PERFORM A DISK OPERATION      
;___________________________________________________________________________________
;                   
DSKOP:
          LD        HL,FLATCH_STORE               ; POINT TO FLATCH
          SET       1,(HL)                        ; SET MOTOR OFF
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER
                                                  ;
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
          CP        0FFH                          ; DID IT RETURN WITH ERROR CODE?
          JP        Z,DSKEXIT           ; #IF YES, EXIT WITH ERROR CODE
                                                  ;         
          LD        A,(UNIT)            ; GET DISK UNIT NUMBER
          AND       03H                           ; MASK FOR FOUR DRIVES.
          LD        B,A                           ; PARK IT IN B
          LD        A,(HEAD)            ; GET HEAD SELECTION
          AND       01H                           ; INSURE SINGLE BIT
          RLA                                     ;
          RLA                                     ; MOVE HEAD TO BIT 2 POSITION
          OR        B                             ; OR HEAD TO UNIT BYTE IN COMMAND BLOCK
          LD        (UNIT),A            ; STORE IN UNIT
                                                  ;
          LD        HL,FLATCH_STORE               ; POINT TO FLATCH
          RES       1,(HL)                        ; SET MOTOR ON
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER        
                                                  ;
          LD        A,03                          ; SPEC#IFY COMMAND
          CALL      PFDATA                        ; PUSH IT
          LD        A,(SRTHUT)                    ; STEP RATE AND HEAD UNLOAD TIME
          CALL      PFDATA                        ; PUSH THAT
          LD        A,(HLT)                       ;
          CALL      PFDATA                        ; PUSH THAT
                                                  ;
          CALL      SETTRACK            ; PERFORM SEEK TO TRACK
                                                  ;
          JP        NZ,DSKEXIT                    ; #IF ERROR, EXIT
                                                  ;
          LD        A,(CMD)                       ; WHAT COMMAND IS PENDING?
          OR        A                             ; SET FLAGS
          JP        DOSO4                         ; NO, MUST BE READ OR WRITE COMMAND
DSKEXIT:  
          LD        HL,FLATCH_STORE               ; POINT TO FLATCH
          SET       1,(HL)                        ; SET MOTOR OFF
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER
                                        ;
          OR        0FFH                          ; SET -1 #IF ERROR
          RET

RESULT:
          LD        C,07H                         ; LOAD C WITH NUMBER OF STATUS BYTES
          LD        HL,ST0                        ; POINT TO STATS STORAGE
RS3:
          CALL      GFDATA                        ; GET FIRST BYTE
          LD        (HL),A                        ; SAVE IT
          INC       HL                            ; POINTER++
          DEC       C                             ; CC-1
          JP        NZ,RS3                        ; LOOP TIL C0
          LD        A,(ST0)                       ; LOAD STS0
          AND       0F8H                          ; MASK OFF DRIVE #
          LD        B,A                           ; PARK IT
          LD        A,(ST1)                       ; LOAD STS1
          OR        B                             ; ACC OR B ->ACC #IF 0 THEN SUCCESS
                                                  ;
RSTEXIT:
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS, MAKE SURE IT IS CLEAR
          LD        HL,FLATCH_STORE               ; POINT TO FLATCH
          SET       1,(HL)                        ; SET MOTOR OFF
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER
                                                  ;
          CALL      DMPDKR                        ;
          RET                                     ; DONE RETURN TO CALLER.
          
          
DOSO4:
                                                  ;
          LD        HL,SECTOR_BUFFER    ; GET BUFFER ADDRESS TO HL
          LD        A,(SECSIZ)                    ; XFERLEN
          LD        C,A                           ; C WILL BE THE NUMBER OF TRANSACTIONS
                                                  ; DIVIDED BY 4
                                                  ;
          LD        A,(CMD)                       ;
          CALL      PFDATA                        ; PUSH COMMAND TO I8272
          LD        A,(UNIT)            ;
          CALL      PFDATA                        ; 
          LD        A,(TRACK)           ;
          CALL      PFDATA                        ; 
          LD        A,(HEAD)            ;
          CALL      PFDATA                        ; 
          LD        A,(SECTOR)                    ;
          CALL      PFDATA                        ; 
          LD        A,(DENS)            ;
          CALL      PFDATA                        ; WHAT DENSITY
          LD        A,(EOTSEC)                    ;
          CALL      PFDATA                        ; ASSUME SC (SECTOR COUNT)  EOT
          LD        A,(GAP)                       ;
          CALL      PFDATA                        ; WHAT GAP IS NEEDED
          LD        A,(DTL)                       ; DTL, IS THE LAST COMMAND BYTE TO I8272
          CALL      PFDATAS
          LD        A,(CMD)                       ; READ IS 0 IS THIS A READ OR WRITE?
          AND       00000001B           ; WRITE IS 1
          JP        NZ,WRR                        ; JMP WRITE #IF 1
;

          ;
; PERFORM READ
; LOOP EXECUTES 4X, THIS ALLOWS C RATHER THAN BC AS COUNTER
; SAVING A FEW TSTATES. MAKES UP TO 1024 BYTE SECTORS POSSIBLE.
; FROM READ TO READ MUST NOT EXCEED 25US WORST CASE MIN.
; (76T STATES FOR 3MHZ 8085) or (100 T STATES FOR 4MHZ Z80)
;

RDD_POLL:
FDC_READP0:
          IN        A,(FMSR)            ;
          OR        A                             ; test if byte ready RQM1
          JP        P,FDC_READP0                  ;         
                                                  ;
          AND       20H                           ;
          JP        Z,DSKOPEND                    ;jump if in results mode
                                                  ;
          IN        A,(FDATA)           ;
          LD        (HL),A                        ;
          INC       HL                            ;

FDC_READP1:
          IN        A,(FMSR)            ;
          OR        A                             ;
          JP        P,FDC_READP1                  ;
                                                  ;
          AND       20H                           ;
          JP        Z,DSKOPEND                    ;
                                                  ;
          IN        A,(FDATA)           ;
          LD        (HL),A                        ;
          INC       HL                            ;
                                                  ;
FDC_READP2:
          IN        A,(FMSR)            ;
          OR        A                             ;
          JP        P,FDC_READP2                  ;
                                                  ;
          AND       20H                           ;
          JP        Z,DSKOPEND                    ;
                                                  ;
          IN        A,(FDATA)           ;
          LD        (HL),A                        ;
          INC       HL                            ;
                                                  ;
FDC_READP3:
          IN        A,(FMSR)            ; 11
          OR        A                             ; 4
          JP        P,FDC_READP3                  ; 10
                                                  ;
          AND       20H                           ; 7
          JP        Z,DSKOPEND                    ; 10
                                                  ;
          IN        A,(FDATA)           ; 11
          LD        (HL),A                        ; 10
          INC       HL                            ; 11
                                                  ;
          DEC       C                             ; 4
          JP        NZ,FDC_READP0                 ; 11

DSKOPEND:
          LD        HL,FLATCH_STORE               ; POINT TO FLATCH
          SET       0,(HL)                        ; SET TC
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER
          NOP                                     ;
          NOP                                     ; 2 MICROSECOND DELAY
          RES       0,(HL)                        ; RESET TC
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER
          NOP                                     ;
          NOP                                     ; 2 MICROSECOND DELAY
          NOP                                     ;
          NOP                                     ; 2 MICROSECOND DELAY
          SET       1,(HL)                        ; TURN OFF MOTOR
          CALL      OUTFLATCH           ; OUTPUT TO CONTROLLER
          JP        RESULT                        ; GET STATUS BYTES <RESULT PHASE>
WRR:
FDC_WRITEP0:
          IN        A,(FMSR)            ;
          OR        A                             ; test if byte ready RQM1
          JP        P,FDC_WRITEP0                 ;         
                                                  ;
          AND       20H                           ;
          JP        Z,DSKOPEND                    ;jump if in results mode
                                                  ;
          LD        A,(HL)                        ;
          OUT       (FDATA),A           ;         
          INC       HL                            ;

FDC_WRITEP1:
          IN        A,(FMSR)            ;
          OR        A                             ;
          JP        P,FDC_WRITEP1                 ;
                                                  ;
          AND       20H                           ;
          JP        Z,DSKOPEND                    ;
                                                  ;
          LD        A,(HL)                        ;
          OUT       (FDATA),A           ;         
          INC       HL                            ;
                                                  ;
FDC_WRITEP2:
          IN        A,(FMSR)            ;
          OR        A                             ;
          JP        P,FDC_WRITEP2                 ;
                                                  ;
          AND       20H                           ;
          JP        Z,DSKOPEND                    ;
                                                  ;
          LD        A,(HL)                        ;
          OUT       (FDATA),A           ;         
          INC       HL                            ;
                                                  ;
FDC_WRITEP3:
          IN        A,(FMSR)            ; 11
          OR        A                             ; 4
          JP        P,FDC_WRITEP3                 ; 10
                                                  ;
          AND       20H                           ; 7
          JP        Z,DSKOPEND                    ; 10
                                                  ;
          LD        A,(HL)                        ;
          OUT       (FDATA),A           ;         
          INC       HL                            ; 11
                                                  ;
          DEC       C                             ; 4
          JP        NZ,FDC_WRITEP0                ; 11
          JP        DSKOPEND            ; 10
          
                    
;__SETTRACK_______________________________________________________________________________ 
;
;         SEEK TO A TRACK ON GIVEN UNIT
;         A: TRACK #
;___________________________________________________________________________________
;
SETTRACK:
          LD        A,(TRACK)           ; GET TRACK
          OR        A                             ; SET FLAGS
          JP        Z,RECAL                       ; #IF 0 PERFORM RECAL INSTEAD OF SEEK
          LD        A,0FH                         ; SEEK COMMAND
          CALL      PFDATA                        ; PUSH COMMAND
          LD        A,(UNIT)            ; SAY WHICH UNIT
          CALL      PFDATA                        ; SEND THAT
          LD        A,(TRACK)           ; TO WHAT TRACK
          CALL      PFDATA                        ; SEND THAT TOO
          JP        WAINT                         ; WAIT FOR INTERRUPT SAYING DONE
RECAL:
          LD        A,07H                         ; RECAL TO TRACK 0
          CALL      PFDATA                        ; SEND IT
          LD        A,(UNIT)            ; WHICH UNIT
          CALL      PFDATA                        ; SEND THAT TOO
;
WAINT:
;
          CALL      DELAYHSEC           ; DELAY TO LET HEADS SETTLE BEFORE READ
                                                  ;
                                                  ; WAIT HERE FOR INTERRPT SAYING DONE
                                                  ; LOOP TIL INTERRUPT
          CALL      CHECKINT            ; CHECK INTERRUPT STATUS
;
          RET

;__PFDATAS_______________________________________________________________________________ 
;
; WRITE A COMMAND OR PARAMETER SEQUENCE
;
; TRANSFERS ARE SYNCHONIZED BYT MSR D7 <RQM> AND D6 <DIO>
;         RQM  DIO
;         0         0         BUSY
;         1         0         WRITE TO DATA REGISTER PERMITTED
;         1         1         BYTE FOR READ BY HOST PENDING
;         0         1         BUSY
;
;___________________________________________________________________________________
;
PFDATAS:
          PUSH      AF                            ; STORE AF
PFDS1:
          IN        A,(FMSR)            ; READING OR WRITING IS KEYS TO D7 RQM
          AND       80H                           ; MASK OFF RQM BIT 
          JP        Z,PFDS1                       ; WAIT FOR RQM TO BE TRUE.
          IN        A,(FMSR)            ; READ STATUS
          AND       40H                           ; WAITING FOR INPUT?
          CALL      NZ,ERRORT           ; NO, SIGNAL ERROR
          POP       AF                            ; RESTORE AF
          OUT       (FDATA),A           ; OUTPUT A TO CONTROLLER
          RET                 
          
;__PFDATA_______________________________________________________________________________ 
;
; WRITE A COMMAND OR PARAMETER SEQUENCE
;
; TRANSFERS ARE SYNCHONIZED BYT MSR D7 <RQM> AND D6 <DIO>
;         RQM  DIO
;         0         0         BUSY
;         1         0         WRITE TO DATA REGISTER PERMITTED
;         1         1         BYTE FOR READ BY HOST PENDING
;         0         1         BUSY
;
;___________________________________________________________________________________
;
PFDATA:
          PUSH      AF                            ; STORE AF
PFD1:
          IN        A,(FMSR)            ; READING OR WRITING IS KEYS TO D7 RQM
          AND       80H                           ; MASK OFF RQM BIT 
          JP        Z,PFD1                        ; WAIT FOR RQM TO BE TRUE.
          IN        A,(FMSR)            ; READ STATUS
          AND       40H                           ; WAITING FOR INPUT?
          CALL      NZ,ERRORT           ; NO, SIGNAL ERROR
          POP       AF                            ; RESTORE AF
          OUT       (FDATA),A           ; OUTPUT A TO CONTROLLER
          NOP                                     ; WAIT 24 US BEFORE READING FMSR AGAIN
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          RET


;__CHECKINT_______________________________________________________________________________ 
;
; CHECK FOR ACTIVE FDC INTERRUPTS BEFORE GIVING I8272 COMMANDS
; POLL RQM FOR WHEN NOT BUSY AND THEN SEND FDC
; SENSE INTERRUPT COMMAND.  #IF IT RETURNS WITH NON ZERO
; ERROR CODE, PASS BACK TO CALLING ROUTINE FOR HANDLING
;___________________________________________________________________________________
;
CHECKINT:
          IN        A,(FMSR)            ; READING OR WRITING IS KEYS TO D7 RQM
          AND       80H                           ; MASK OFF RQM BIT
          JP        Z,CHECKINT                    ; WAIT FOR RQM TO BE TRUE. WAIT UNTIL DONE
          IN        A,(FMSR)            ; READ STATUS
          AND       40H                           ; WAITING FOR INPUT?
          CALL      NZ,CHECKINTDONE               ; NO, SIGNAL ERROR
          CALL      SENDINT                       ; SENSE INTERRUPT COMMAND
CHECKINTDONE:
          RET                                     ;
          

;__DELAYHSEC_______________________________________________________________________________ 
;
; DELAY FOR 1/2 SECOND
;___________________________________________________________________________________
;                   
DELAYHSEC:
          LD        HL,00000H           ; 65536
DELDM:
          NOP                                     ; (4 T) 
          NOP                                     ; (4 T)
          NOP                                     ; (4 T)
          NOP                                     ; (4 T)
          DEC       L                             ; (6 T)
          JP        NZ,DELDM            ; (10 T) 24 T  8 MICROSECONDS AT 4 MHZ
          DEC       H                             ; (6 T)
          JP        NZ,DELDM            ; (10 T) (8 US * 256) * 256  524288 US  .5 SECONDS
          RET

;__ERRORT_______________________________________________________________________________ 
;
; ERROR HANDLING
;___________________________________________________________________________________
;                             
ERRORT:
          IN        A,(FDATA)           ; CLEAR THE JUNK OUT OF DATA REGISTER
          IN        A,(FMSR)            ; CHECK WITH RQM
          AND       80H                           ; #IF STILL NOT READY, READ OUT MORE JUNK
          JP        Z,ERRORT            ;
          LD        A,0FFH                         ; RETURN ERROR CODE -1
                                                  ;
          RET
;__SENDINT_______________________________________________________________________________ 
;
; SENSE INTERRUPT COMMAND
;___________________________________________________________________________________
;                             
SENDINT:
          LD        A,08H                         ; SENSE INTERRUPT COMMAND
          CALL      PFDATA                        ; SEND IT
          CALL      GFDATA                        ; GET RESULTS
          LD        (ST0A),A            ; STORE THAT
          AND       0C0H                          ; MASK OFF INTERRUPT STATUS BITS
          CP        80H                           ; CHECK #IF INVALID COMMAND
          JP        Z,ENDSENDINT                  ; YES, EXIT
          CALL      GFDATA                        ; GET ANOTHER (STATUS CODE 1)
          LD        (ST1A),A            ; SAVE THAT
          LD        A,(ST0A)            ; GET FIRST ONE
          AND       0C0H                          ; MASK OFF ALL BUT INTERRUPT CODE 00 IS NORMAL
ENDSENDINT:
          RET                                     ;ANYTHING #ELSE IS AN ERROR


;__GFDATA_______________________________________________________________________________ 
;
; GET DATA FROM FLOPPY CONTROLLER
;
; TRANSFERS ARE SYNCHONIZED BYT MSR D7 <RQM> AND D6 <DIO>
;         RQM  DIO
;         0         0         BUSY
;         1         0         WRITE TO DATA REGISTER PERMITTED
;         1         1         BYTE FOR READ BY HOST PENDING
;         0         1         BUSY
;
;___________________________________________________________________________________
;                   
GFDATA:
          IN        A,(FMSR)            ; READ STATUS BYTE
          AND       80H                           ; MASK OFF RQM
          JP        Z,GFDATA            ; LOOP WHILE BUSY
          IN        A,(FMSR)            ; READ STSTUS BUTE
          AND       40H                           ; MASK OFF DIO
          CALL      Z,ERRORT            ; #IF WRITE EXPECTED RUN ERRORRT
          IN        A,(FDATA)           ; READ DATA
          NOP                                     ; WAIT 24 US BEFORE READING FMSR AGAIN
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          NOP                                     ;
          RET
          ENDIF

;-----------------------------------------------------------------------------

;         INITIALIZED DATA AREA

OUTCNT:   DEFB        1                   ;RESERVE ONE BYTE FOR COUNTER
WRMFLG:   DEFB        0                   ;WARM BOOT FLAG


;         TEXT STRINGS

TXT_RO:   DEFB        CR,LF
          DEFM        "R/O"
          DEFB        ENDS

TXT_SO:   DEFM        "CP/M V2.2C "

          DEFB        NK/10+'0', NK MOD 10+'0'
          DEFM        "k - V09 "
          DEFB        CR,LF
          DEFB        ENDS

TXT_WARM:
          DEFB        CR,LF
          DEFM        "WARM BOOT - "
          DEFB        ENDS

TXT_COLD:
          DEFB        CR,LF
          DEFM        "COLD BOOT - "
          DEFB        ENDS

          IF        DEBUG     
TXT_RED:
          DEFM        "R-"
          DEFB        ENDS

TXT_WRT:
          DEFM        "W-"
          DEFB        ENDS

TXT_HED:
          DEFM        " H"
          DEFB        ENDS
          
TXT_DRV:
          DEFM        " D"
          DEFB        ENDS
          
TXT_TRK:
          DEFM        " T"
          DEFB        ENDS

TXT_SEC:
          DEFM        " S"
          DEFB        ENDS

TXT_PAG:
          DEFM        " P"
          DEFB        ENDS

TXT_ADR:
          DEFM        " O"
          DEFB        ENDS

TXT_SPC:
          DEFB        " "
          DEFB        ENDS

TXT_CL:
          DEFB        CR,LF
          DEFB        ENDS
          ENDIF

;-----------------------------------------------------------------------------
;
; DISK COMMAND BLOCK
;
CMD:      DEFB        0                             ; COMMAND READ OR WRITE,
UNIT:     DEFB        0                             ; PHYSICAL DRIVE 0->3
HEAD:     DEFB        0                             ; HEAD SEL 0 OR 1
DENS:     DEFB        2                             ; DENSITY
EOTSEC:   DEFB        09                            ; LAST SECTOR OF TRACK
GAP:      DEFB        1BH                           ; VALUE FOR IRG <GAP3>
SECSIZ:   DEFB        080H                          ; HOW MANY BYTES TO TRANSFER/4
DTL:      DEFB        0FFH                          ; SIZE OF SECTOR
SRTHUT:   DEFB        7FH                           ; STEP RATE AND HEAD UNLOAD TIME
HLT:      DEFB        05H                           ; HEAD LOAD TIME
MIN:      DEFB        MINI                          ; LATCH BIT PATTERN FOR FDC9229 MINITRUE
PRE:      DEFB        PRECOMP                       ; LATCH BIT PATTERN FOR FDC9229 PRECOMP125NS
;
; FLOPPY STATUS RESULT STORAGE
;
ST0:      DEFB        0                             ; STORE STATUS 0
ST1:      DEFB        0                             ; ST1
ST2:      DEFB        0                             ; ST2
SCYL:     DEFB        0                             ; TRACK
SHEAD:    DEFB        0                             ; HEAD 0 OR 1
SREC:     DEFB        0                             ; SECTOR
SNBIT:    DEFB        0                             ; DENSITY
ST0A:     DEFB        0                             ; STORE STATUS 0
ST1A:     DEFB        0                             ; ST1
RETRY:    DEFB        0                             ; RETRIES

FLATCH_STORE:
          DEFB        00;-----------------------------------------------------------------------------

;         UNINITIALIZED DATA AREA

TRACK:    DEFS        2                   ;TWO BYTES FOR TRACK #
PAGE:     DEFS        1                   ;COPY OF PAGE BYTE
PAGOFF:   DEFS        2                   ;OFFSET OF SECTOR IN PAGE
SECTOR:   DEFS        2                   ;TWO BYTES FOR SECTOR #
V_SECTOR:
          DEFS        2                   ;TWO BYTES FOR VIRTUAL SECTOR #
OFFSET:   DEFS        2                   ;SECTOR OFFSET IN TRACK
DMAADR:   DEFS        2                   ;DIRECT MEMORY ADDRESS
DRIVE:    DEFS        1                   ;DISK NUMBER 0-15
DPHSTO:   DEFS        2                   ;DPH ADDRESS FOR CURRENT DRIVE
DPBSTO:   DEFS        2                   ;DPB ADDRESS FOR CURRENT DRIVE
LOGREQ:   DEFS        1                   ;LOGON REQUEST FLAG

          IF        HARDDRV                       
LBA_OFFSET_LO:
          DEFS        2                   ;IDE HD PARTITION STARTING SECTOR (LOW 16 BITS)
LBA_OFFSET_HI:
          DEFS        2                   ;IDE HD PARTITION STARTING SECTOR (HI 16 BITS, 12 USED)
LBA_TARGET_LO:
          DEFS        2                   ;IDE HD PARTITION TARGET SECTOR (LOW 16 BITS)
LBA_TARGET_HI:
          DEFS        2                   ;IDE HD PARTITION TARGET SECTOR (HI 16 BITS, 12 USED)
IDE_LBA0:
          DEFS        1                   ;SET LBA 0:7
IDE_LBA1:
          DEFS        1                   ;SET LBA 8:15
IDE_LBA2:
          DEFS        1                   ;SET LBA 16:23
IDE_LBA3:
          DEFS        1                   ;LOWEST 4 BITS USED ONLY TO ENABLE LBA MODE 
SECTOR_INDEX:
          DEFS        1                   ;WHERE 128 BYTE CP/M SECTOR IS IN 512 BYTE IDE HD SECTOR
SECTOR_BUFFER:
          DEFS        512                 ;STORAGE FOR 512 BYTE IDE HD SECTOR

TMPBUF:   EQU SECTOR_BUFFER
          ELSE
TMPBUF:   DEFS        128                 ;TEMPORARY BUFFER FOR DISK TRANSFERS
          ENDIF

;         SCRATCH RAM AREA FOR BDOS USE

DIRBF:    DEFS        128                 ;SCRATCH DIRECTORY AREA
;         ALLOCATION VECTORS - (DSM+1/8)
ALL00:    DEFS        127                 ;ALLOCATION VECTOR 0 1013/8
ALL01:    DEFS        55                  ;ALLOCATION VECTOR 1 (437/8)
          IF        HARDDRV
ALL02:    DEFS        255                 ;ALLOCATION VECTOR 2 (2041/8)
ALL03:    DEFS        255                 ;ALLOCATION VECTOR 3 (2041/8)
ALL04:    DEFS        255                 ;ALLOCATION VECTOR 4 (2041/8)
ALL05:    DEFS        255                 ;ALLOCATION VECTOR 4 (2041/8)
          ENDIF

          IF        FLOPDRV
ALL06:    DEFS        35                  ;ALLOCATION VECTOR 4 (360/8)
          ENDIF

CHK00:    DEFS        0                   ;NOT USED FOR FIXED MEDIA
CHK01:    DEFS        0                   ;NOT USED FOR FIXED MEDIA
          IF        HARDDRV
CHK02:    DEFS        0                   ;NOT USED FOR FIXED MEDIA
CHK03:    DEFS        0                   ;NOT USED FOR FIXED MEDIA
CHK04:    DEFS        0                   ;NOT USED FOR FIXED MEDIA
CHK05:    DEFS        0                   ;NOT USED FOR FIXED MEDIA
          ENDIF

          IF        FLOPDRV
CHK06:    DEFS        32                  ;360KB FLOPPY
          ENDIF

;-----------------------------------------------------------------------------
          IF        ENDFIL
          ORG       BIOSO+012FFH
          DEFB        055H
          ENDIF

          END
